/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.vuid.rest.session;

import java.io.IOException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import gov.vha.isaac.ochre.api.User;
import gov.vha.isaac.ochre.api.UserRole;
import gov.vha.isaac.ochre.api.bootstrap.TermAux;
import gov.vha.isaac.ochre.api.util.UuidT5Generator;

/**
 * 
 * {@link UserServiceUtils}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class UserServiceUtils {
	
	private static Logger log = LogManager.getLogger(UserServiceUtils.class);
	
	static User getUserFromJson(String jsonToUse) throws JsonParseException, JsonMappingException, IOException {
		ObjectMapper mapper = new ObjectMapper();
		Map<?, ?> map = null;
		map = mapper.readValue(jsonToUse, Map.class);
		
		String userName = (String)map.get("user");
		Set<UserRole> roleSet = new HashSet<>();
		Collection<?> roles = (Collection<?>)map.get("roles");
		for (Object roleMapObject : roles) {
			Map<?,?> roleMap = (Map<?,?>)roleMapObject;
			String roleName = (String)roleMap.get("name");
			
			roleSet.add(UserRole.safeValueOf(roleName).get());
		}
		
		final UUID uuidFromUserFsn = UserServiceUtils.getUuidFromUserName(userName);;
	
		User newUser = new User(userName, uuidFromUserFsn, roleSet);
		
		return newUser;
	}

	static UUID getUuidFromUserName(String userName) {
		return UuidT5Generator.get(TermAux.USER.getPrimordialUuid(), userName);
	}

	static User getUserFromUrl(URL url, String ssoToken) throws JsonParseException, JsonMappingException, IOException {
//		/*
//		 * Example URL for get_roles_by_token
//		 * URL url = new URL("https://DNS                     token=" + token);
//		 */
//		/*
//		 * Example SSO Token
//		 * %5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx94%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFXk%5Cx80%5CxB1%5CxA3%5CxF3%5Cx8D%5CxB1%5Cx7F%5CxBC%5Cx02K%22%2C+%22k%5Cf%5CxDC%5CxF7%2CP%5CxB2%5Cx97%5Cx99%5Cx99%5CxE0%5CxE1%7C%5CxBF%5Cx1DK%22%2C+%22J%5Cf%5Cx9B%5CxD8w%5Cx15%5CxFE%5CxD3%5CxC7%5CxDC%5CxAC%5Cx9E%5Cx1C%5CxD0bG%22%5D
//		 */
//		//String json = "{\"roles\":[{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}],\"token_parsed?\":true,\"user\":\"DNS  \",\"type\":\"ssoi\",\"id\":10005}";

		log.info("Attempting to read user from url '{}'", url);
		Map<String, String> params = new HashMap<>();	
		params.put("token", URLEncoder.encode(ssoToken, "UTF-8"));
		String jsonResultString = PrismeServiceUtils.getResultJsonFromPrisme(PrismeServiceUtils.getTargetFromUrl(url), url.getPath(), params);
		
		return UserServiceUtils.getUserFromJson(jsonResultString);
	}
}
